#!/usr/bin/env python3
"""
Extract text from a screenshot or image using OCR.

Usage:
    python screenshot-ocr.py <image_file>
    python screenshot-ocr.py screenshot.png > extracted.md

Requirements:
    pip install pytesseract pillow

    Also need Tesseract installed:
    - Mac: brew install tesseract
    - Ubuntu: apt-get install tesseract-ocr
"""

import sys

try:
    import pytesseract
    from PIL import Image
except ImportError:
    print("Error: Required packages not installed")
    print("Run: pip install pytesseract pillow")
    print("Also install Tesseract: brew install tesseract (Mac)")
    sys.exit(1)


def extract_text(image_path):
    """Extract text from image using OCR."""
    try:
        image = Image.open(image_path)
        text = pytesseract.image_to_string(image)
        return text.strip()
    except Exception as e:
        return f"Error: {e}"


def main():
    if len(sys.argv) < 2:
        print("Usage: python screenshot-ocr.py <image_file>")
        sys.exit(1)

    image_path = sys.argv[1]

    print(f"# OCR Extraction\n")
    print(f"**Source:** {image_path}\n")
    print("---\n")

    text = extract_text(image_path)
    print(text)


if __name__ == "__main__":
    main()
