#!/usr/bin/env python3
"""
Convert a CSV file to a markdown table.

Usage:
    python csv-to-markdown.py <file.csv>
    python csv-to-markdown.py data.csv > table.md

Options:
    --max-rows N    Limit output to N rows (default: all)
    --truncate N    Truncate cell content to N characters (default: 50)
"""

import sys
import csv
import argparse


def csv_to_markdown(filepath, max_rows=None, truncate=50):
    """Convert CSV to markdown table format."""

    with open(filepath, 'r', encoding='utf-8') as f:
        reader = csv.reader(f)
        rows = list(reader)

    if not rows:
        return "Empty CSV file"

    # Get headers and data
    headers = rows[0]
    data = rows[1:]

    if max_rows:
        data = data[:max_rows]

    # Truncate helper
    def trunc(text):
        text = str(text).replace('\n', ' ').replace('|', '\\|')
        if len(text) > truncate:
            return text[:truncate-3] + '...'
        return text

    # Build table
    lines = []

    # Header row
    header_line = '| ' + ' | '.join(trunc(h) for h in headers) + ' |'
    lines.append(header_line)

    # Separator
    separator = '| ' + ' | '.join('---' for _ in headers) + ' |'
    lines.append(separator)

    # Data rows
    for row in data:
        # Pad row if needed
        while len(row) < len(headers):
            row.append('')
        row_line = '| ' + ' | '.join(trunc(cell) for cell in row[:len(headers)]) + ' |'
        lines.append(row_line)

    return '\n'.join(lines)


def main():
    parser = argparse.ArgumentParser(description='Convert CSV to Markdown table')
    parser.add_argument('file', help='CSV file to convert')
    parser.add_argument('--max-rows', type=int, help='Maximum rows to output')
    parser.add_argument('--truncate', type=int, default=50, help='Truncate cells to N chars')

    args = parser.parse_args()

    try:
        result = csv_to_markdown(args.file, args.max_rows, args.truncate)
        print(result)
    except FileNotFoundError:
        print(f"Error: File not found: {args.file}")
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)


if __name__ == "__main__":
    main()
